﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpSvn;
using RecoderProject;
using System.Windows.Forms;
using RevisionAnalyser.Global;

namespace RevisionAnalyser.Projects.RecoderProject.Tasks
{
    public class AddTask : RecoderTask
    {
        private RecoderProject _project = (RecoderProject)CurrentProject.Instance.Project;
        private long _id;
        private string _author;
        private DateTime _time;
        private string _logMessage;

        public AddTask(long id, string author, DateTime time, string logMessage)
        {
            _id = id;
            _author = author;
            _time = time;
            _logMessage = logMessage;
        }

        public override void Run()
        {
            try
            {
                UpdateSvn();
                AnalyseCode();
                _project.RevisionSet.AddRevision(new SvnRevision(_id, _author, _time, _logMessage), false);
                AddLog(String.Format("Added SVN revision {0}", _id), Constants.COLOR_GREEN);
            }
            catch (Exception ex)
            {
                AddLog("Error: " + ex.Message, Constants.COLOR_RED);
            }
            finally
            {
                TaskFinished();
            }
        }

        private void UpdateSvn()
        {
            using (SvnClient client = _project.GetSvnClient())
            {
                client.Processing += new EventHandler<SvnProcessingEventArgs>(client_Processing);
                client.Notify += new EventHandler<SvnNotifyEventArgs>(client_Notify);
                SvnCheckOutArgs args = new SvnCheckOutArgs();
                args.Revision = new SharpSvn.SvnRevision(_id);
                client.CheckOut(new SvnUriTarget(_project.SvnUrl), _project.GetSvnFolder(), args);
            }
        }

        private void AnalyseCode()
        {
            AddLog("Analysing code", Constants.COLOR_GREEN);

            RecoderAnalyser analyser = GetRecoderAnalyser(_project);
            EnableSubProgressBar(true);
            bool ignoreErrors = false;
            bool succesful = false;

            while (!succesful)
            {
                succesful = analyser.AnalyseRevision(_project.GetSvnFolder(), String.Format("{0}\\Dataset.bin", _project.GetAnalysedFolder()), _id, _project.JarLocations, ignoreErrors);
                if (!succesful)
                {
                    if (MessageBox.Show("Do you want to add/remove libraries? \nNote: When clicking 'No' the errors will be ignored.", "Analyse error(s) found!", MessageBoxButtons.YesNo, MessageBoxIcon.Question).Equals(DialogResult.Yes))
                    {                
                        GetTaskForm().Invoke((Action)(() => { _project.OpenLibrarySettings(); }));
                        AddLog("User action: add/remove libraries", Constants.COLOR_BLUE);
                    }
                    else
                    {
                        ignoreErrors = true;
                        AddLog("User action: ignore errors enabled", Constants.COLOR_BLUE);
                    }
                }
            }
            EnableSubProgressBar(false);
        }

        void client_Notify(object sender, SvnNotifyEventArgs e)
        {
            AddLog(String.Format("{0}: {1}", e.Action.ToString().Remove(0, "Update".Length), e.FullPath), Constants.COLOR_GREEN);
        }

        private void client_Processing(object sender, SvnProcessingEventArgs e)
        {
            if (e.CommandType.ToString().Equals("CheckOut"))
            {
                this.AddLog("Checkout/Update started", Constants.COLOR_GREEN);
            }
        }
    }
}
